/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***********************************************************************
 * File name     : board.c
 * board properties and generic functions
 ***********************************************************************/


#ifdef BEST_FIRMWARE

#include <capitype.h>

#else

#include <typedefs.h>

/* TODO: remove this as soon as puprpget has found a good place */
#include <mini_api.h>


#endif

#include <board.h>
#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regconst.h>
#include <regx11.h>
#include <session.h>
#include <tardefs.h>


/* -----------------------------------------------------------------
 * Retrieve the Status Register
 * ----------------------------------------------------------------- */

b_errtype EXPORT BestStatusRegGet(
    b_handletype handle,
    b_int32 * value_ptr
)
{
  B_DECLARE_FUNCNAME("BestStatusRegGet [sregget]");
  B_TRY_VARS_NO_PROG;

  B_FCT_PARAM_NULL_POINTER_CHK(value_ptr);

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    
    /* need capi license OR hostint license! */
    if (BestCapabilityCheck(handle, B_CAPABILITY_HOSTINT) != B_E_OK)
    {
      B_TRY_LICENSE(B_CAPABILITY_CAPI);
    }

    if (Best16BitRegisterFile(handle))
    {

#endif
      
      B_TRY(BBCReceiveLong(handle, CMD_SYS_STATUS_GET, value_ptr));

#ifndef BEST_FIRMWARE
      
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, BEST_STATUS_MINI, (b_int8 *) value_ptr, 4UL));
    }

#endif
    
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * Send a 1 in any bit-position to clear that bit in the status register.
 * Note that the firmware will ignore any attempts to clear a read-only bit.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestStatusRegClear(
    b_handletype handle,
    b_int32 clearpattern
)
{
  B_DECLARE_FUNCNAME ("BestStatusRegClear [sregclr]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    
    /* need capi license OR hostint license! */
    if (BestCapabilityCheck(handle, B_CAPABILITY_HOSTINT) != B_E_OK)
    {
      B_TRY_LICENSE(B_CAPABILITY_CAPI);
    }

    if (Best16BitRegisterFile(handle))
    {

#endif
      
      B_TRY(BBCSendLong(handle, CMD_SYS_STATUS_CLR, &clearpattern));

#ifndef BEST_FIRMWARE
      
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, BEST_STATUS_MINI, (b_int8 *) & clearpattern, 4UL));
    }

#endif
    
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestBoardPropSet(
    b_handletype handle,
    b_boardproptype boardprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestBoardPropSet [bdprpset]");
  B_TRY_VARS_NO_PROG;
  b_int32 zw;
  b_int8 zw8;
  B_TRY_BEGIN
  {
    /* no license check */
    B_TRY(BestParamCheck(handle, B_PARAM_BOARD, (b_int32)boardprop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_BOARD_PROP_SET,
          (b_int8) boardprop, value));
    }

    else
    {
      switch (boardprop)
      {

#ifndef BEST_FIRMWARE
	
      case B_BOARD_RSTMODE:
        zw8 = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, BOARD_RSTMODE, &zw8, 1UL));
        break;

      case B_BOARD_PERREN:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
            (b_int8ptr) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));
        if (value == 0)
        {
          zw &= 0xffffffbfUL;
        }

        else
        {
          zw |= 0x00000040;
        }

        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));
        break;

      case B_BOARD_SERREN:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
            (b_int8ptr) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));

        if (value == 0)
        {
          zw &= 0xfffffeffUL;
        }

        else
        {
          zw |= 0x00000100;
        }

        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));
        break;

      case B_BOARD_ROMUSAGE:
        zw8 = (b_int8) value;

        if (value == B_ROMUSAGE_INTERNAL)
        {
          /* check for ROM decoder diabled */
          zw = 0x00000030UL;
          B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
              (b_int8ptr) & zw, 4UL));
          B_TRY(BestBasicBlockRead(handle, CONF_SPACE_MASK,
              (b_int8ptr) & zw, 4UL));

          if ((zw & 0xFFFFFFFEUL) != 0)
          {
            B_TRY_ERROR(B_E_ROM_ENABLED);
          }
        }

        B_TRY(BestBasicBlockWrite(handle, EXP_ROM_MODE,
            (b_int8ptr) & zw8, 1UL));
        break;

#endif
	
      default:
        B_FCT_PARAM_ERROR(boardprop, "Invalid Property");
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}

b_errtype EXPORT BestBoardPropGet(
    b_handletype handle,
    b_boardproptype boardprop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestBoardPropGet [bdprpget]");

  B_TRY_VARS_NO_PROG;
  b_int32 zw;
  b_int8 zw8;
  B_FCT_PARAM_NULL_POINTER_CHK(value);

  B_TRY_BEGIN
  {
    /* no license checking */

    /* Check range of boardprop */
    B_TRY(BestParamProbe(handle, B_PARAM_BOARD, (b_int32)boardprop));

    if (Best16BitRegisterFile(handle))
    {
      switch (boardprop)
      {
      case B_BOARD_ROMUSAGE:
        *value = B_ROMUSAGE_EXTERNAL;
        break;

      default:
        B_TRY(BestAbstractPropLongGet(handle, CMD_BOARD_PROP_GET,
            (b_int8) boardprop, value));
      }                         /*lint !e788 not all enums used */
    }

    else
    {
      switch (boardprop)
      {

#ifndef BEST_FIRMWARE
	
      case B_BOARD_RSTMODE:
        B_TRY(BestBasicBlockRead(handle, BOARD_RSTMODE,
            (b_int8ptr) & zw8, 1UL));
        *value = (b_int32) zw8;
        break;

      case B_BOARD_PERREN:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
            (b_int8ptr) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));

        *value = ((zw & 0x00000040) == 0) ? 0 : 1;
        break;

      case B_BOARD_SERREN:
        zw = 0x00000004;
        B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
            (b_int8ptr) & zw, 4UL));
        B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA,
            (b_int8ptr) & zw, 4UL));

        *value = ((zw & 0x00000100) == 0) ? 0 : 1;
        break;

      case B_BOARD_ROMUSAGE:
        B_TRY(BestBasicBlockRead(handle, EXP_ROM_MODE, &zw8, 1UL));
        *value = (b_int32) zw8;
        break;

#endif
	
      default:
        B_FCT_PARAM_ERROR(boardprop, "Invalid Property");
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* sets all generic properies to their specified defautl value */
b_errtype EXPORT BestBoardPropDefaultSet(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i;
  b_int32 conf_restore;
  b_generic_infotype * GenInfo;
  b_param_infotype * ParamInfo;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_BOARD, &GenInfo));
    B_TRY(BestPowerUpPropGet(handle, B_PU_CONFRESTORE, &conf_restore));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(
        handle, B_PARAM_BOARD, i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

      switch (ParamInfo->proptyp.boardprop)
      {
      case B_BOARD_PERREN:
      case B_BOARD_SERREN:
        if (conf_restore == 0)
          break;

      default:  /*lint !e616 deliberate flow into default */
        /* Set it to default */
        B_TRY(BestBoardPropSet(
          handle, ParamInfo->proptyp.boardprop, ParamInfo->defaultval));
      } /*lint !e788 not all enums used */
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif

/*----------------------------------------------------------------------
 * store settings directly into pu-database
 *----------------------------------------------------------------------*/
b_errtype EXPORT BestBoardStoredPropGet(
    b_handletype handle,
    b_boardproptype boardprop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestBoardStoredPropGet [bdstoredprpget]");
  B_TRY_VARS_NO_PROG;
  B_TRY_BEGIN
  {
    /* no license check */
    B_TRY(BestParamProbe(handle, B_PARAM_BOARD, (b_int32)boardprop));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongGet(handle, CMD_BOARD_STOREDPROP_GET,
          (b_int8) boardprop, value));
    }

    else
    {
      B_TRY_FAIL (B_E_NOT_E2925A);
    }
  }

  B_ERRETURN(B_TRY_RET);
}

/*----------------------------------------------------------------------
 * store settings directly into pu-database
 *----------------------------------------------------------------------*/
b_errtype EXPORT BestBoardPropStore(
    b_handletype handle,
    b_boardproptype boardprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestBoardPropStore [bdprpstore]");
  B_TRY_VARS_NO_PROG;
  B_TRY_BEGIN
  {
    /* no license check */
    B_TRY(BestParamCheck(handle, B_PARAM_BOARD, (b_int32)boardprop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_BOARD_PROP_STORE,
          (b_int8) boardprop, value));
    }

    else
    {
      B_TRY_FAIL (B_E_NOT_E2925A);
    }
  }

  B_ERRETURN(B_TRY_RET);
}



/*----------------------------------------------------------------------
 *  read system info
 *----------------------------------------------------------------------*/

b_errtype EXPORT BestSystemInfoGet(
    b_handletype handle,
    b_systeminfotype infoprop,
    b_int32 *value
)
{
  B_DECLARE_FUNCNAME("BestSystemInfoGet [siget]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {
    /* TODO: range checking of infoprop */
    B_TRY_FCT_PARAM_NULL_POINTER(value);
    B_TRY_FAIL (BestHasIchiban (handle) ? B_E_OK : B_E_NOT_E2925A);
    /* Get current value */
    B_TRY (BestAbstractPropLongGet (handle, CMD_SYSTEM_INFO_GET, 
				    (b_int8)infoprop, value));
  }

  B_ERRETURN (B_TRY_RET);
}
